/**
* @file xlt_trade_struct.h
 * @author 上海前路有光数字科技
 * @date 2026-01-10
 * @brief 本文件定义XLT交易数据结构体
 */

#pragma once

#include "xlt_data_type.h"

namespace x1{
#pragma  pack(push, 8)
/**
 * @brief 新订单请求数据结构
 */
typedef struct xlt_order_insert_info_t{
    uint64_t order_sequence;          ///< 订单序号，当API配置中开启自定义模式时，由用户填写，session内不能重复，否则由系统生成，取值范围[1, 2^20-1]。
    price_t price;                    ///< 价格放大10000倍
    quantity_t quantity;              ///< 数量
    char security_code[CONST_STR_SECURITY_CODE_LEN]; ///< 证券代码，不带空格，并以'\\0'结尾
    ExchangeIndex exchange_index;     ///< 交易市场
    OrderType order_type;             ///< 报单价格类型
    Side side;                        ///< 买卖方向
    BusinessType business_type;       ///< 业务类型
} xlt_order_insert_info_t;

/**
 * @brief 撤单请求数据结构
 */
typedef struct xlt_cancel_order_info_t{
    uint64_t order_sequence;          ///< 订单序号，当API配置中开启自定义模式时，由用户填写，session内不能重复，否则由系统生成，取值范围[1, 2^20-1]。
    uint64_t origin_xid;              ///< 原始订单标识
} xlt_cancel_order_info_t;
#pragma pack(pop)


#pragma pack(push, 1)

/**
 * @brief 订单结束通知
 */
typedef struct {
    uint8_t client_id;                   ///< 委托源标识，用户登录时自定义
    uint64_t sequence;                   ///< 消息序号
    char report_id[CONST_REPORT_ID_LEN]; ///< 回报标识，32字节，非字符串
    uint64_t xid;                        ///< 报单标识
    uint32_t report_num;                 ///< 订单的回报数量
} xlt_order_end_t;

/**
 * @brief 新订单申报成功响应
 */
typedef struct {
    uint8_t client_id;                            ///< 委托源标识，用户登录时自定义
    uint64_t sequence;                            ///< 消息序号(0为快速回报)
    char report_id[CONST_REPORT_ID_LEN];          ///< 回报标识，32字节，非字符串
    uint64_t xid;                                 ///< 报单标识
    OrderStatus order_status;                     ///< 订单状态
    char security_code[CONST_STR_SECURITY_CODE_LEN]; ///< 证券代码，不带空格，并以'\\0'结尾
    ExchangeIndex exchange_index;                 ///< 交易市场
    price_t price;                                ///< 委托价格，放大10000倍
    quantity_t quantity;                          ///< 委托数量
    Side side;                                    ///< 买卖方向
    quantity_t leaves_quantity;                   ///< 剩余数量
    quantity_t cancel_quantity;                   ///< 撤单数量
    int cancel_reason_id;                         ///< 撤单原因id
} xlt_order_response_t;

/**
 * @brief 撤单申报成功响应
 */
typedef struct {
    uint8_t client_id;                            ///< 委托源标识，用户登录时自定义
    uint64_t sequence;                            ///< 消息序号(0为快速回报)
    char report_id[CONST_REPORT_ID_LEN];          ///< 回报标识，32字节，非字符串
    uint64_t xid;                                 ///< 报单标识
    uint64_t origin_xid;                          ///< 原始订单标识（撤单申报时有意义）
    OrderStatus origin_order_status;              ///< 原始订单状态 (部撤/全撤)
    quantity_t cancel_quantity;                   ///< 撤单数量
    int cancel_reason_id;                         ///< 撤单原因id
} xlt_cancel_response_t;

/**
 * @brief 成交回报
 */
typedef struct {
    uint8_t client_id;                      ///< 委托源标识，用户登录时自定义
    uint64_t sequence;                      ///< 消息序号(0为快速回报)
    char report_id[CONST_REPORT_ID_LEN];    ///< 回报标识，32字节，非字符串
    uint64_t xid;                           ///< 报单标识
    OrderStatus order_status;               ///< 订单状态
    price_t price;                          ///< 成交价格，该笔成交回报的成交价格，放大10000倍
    quantity_t quantity;                    ///< 成交数量，该笔成交回报的成交数量
    quantity_t leaves_quantity;             ///< 剩余数量
    int64_t trade_time;                     ///< 成交时间，格式为YYYYMMDDHHMMSSsss
} xlt_trade_report_t;

/**
 * @brief 申报失败响应（新订单申报失败，撤单失败）
 */
typedef struct {
    uint8_t client_id;                      ///< 委托源标识，用户登录时自定义
    uint64_t sequence;                      ///< 消息序号(0为快速回报)
    char report_id[CONST_REPORT_ID_LEN];    ///< 回报标识，32字节，非字符串
    uint64_t xid;                           ///< 报单标识
    uint64_t origin_xid;                    ///< 原始订单标识（撤单申报时有意义）
    xlt_error_info_t error_info;            ///< 错误信息
} xlt_order_error_response_t;

/**
 * @brief 订单信息结构体
 */
typedef struct {
    uint8_t client_id;                ///< 委托源标识，用户登录时自定义
    uint8_t is_cancel_flag;           ///< 撤单标识，如果为1, 当前记录为撤单，否则为新订单。
    uint64_t xid;                     ///< 报单标识
    uint64_t origin_xid;              ///< 原始订单标识
    char cl_order_id[CONST_CL_ORD_ID_LEN];  ///< 报盘合同号
    char origin_cl_order_id[CONST_CL_ORD_ID_LEN]; ///< 如果非空，则代表当前为撤单记录，取值为原委托的cl_order_id(不包含字符串终止符\\0)
    OrderStatus order_status;         ///< 订单状态
    char security_code[CONST_STR_SECURITY_CODE_LEN]; ///< 证券代码
    ExchangeIndex exchange_index;     ///< 交易市场
    OrderType order_type;             ///< 报单价格类型
    Side side;                        ///< 买卖方向
    BusinessType business_type;       ///< 业务类型
    price_t order_price;              ///< 委托价格
    quantity_t order_quantity;        ///< 委托数量
    amount_t order_amount;            ///< 委托金额
    amount_t order_fee;               ///< 委托费用
    quantity_t trade_quantity;        ///< 成交数量
    amount_t trade_amount;            ///< 成交金额
    amount_t trade_fee;               ///< 成交费用
    amount_t trade_fix_fee;           ///< 成交固定费用
    uint32_t trade_count;             ///< 成交笔数
    quantity_t leaves_quantity;       ///< 剩余数量
    quantity_t cancel_quantity;       ///< 撤单数量
    int64_t insert_time;              ///< 委托时间，格式为YYYYMMDDHHMMSSsss
    int64_t trade_time;               ///< 最后一笔成交时间，格式为YYYYMMDDHHMMSSsss
    int64_t cancel_time;              ///< 撤单时间，撤单成功才填写，即成剩撤的单也需要填写。格式为 YYYYMMDDHHMMSSsss
    int64_t update_time;              ///< 最后更新时间，格式为 YYYYMMDDHHMMSSsss
    xlt_error_info_t error_info;      ///< 错误信息
} xlt_order_info_t;

/**
 * @brief 持仓结构体
 */
typedef struct {
    char security_code[CONST_STR_SECURITY_CODE_LEN]; ///< 证券代码
    PositionSecurityType position_security_type; ///< 持仓类型(区分股份是否为配售)
    ExchangeIndex exchange_index;        ///< 交易市场
    quantity_t initial_quantity;         ///< 日初持仓数量
    quantity_t total_quantity;           ///< 总持仓
    quantity_t sellable_quantity;        ///< 可卖持仓
    quantity_t frozen_quantity;          ///< 卖出冻结数量
} xlt_position_info_t;

/**
 * @brief 账户资金信息
 */
typedef struct {
    amount_t available_amount;      ///< 可用资金
    amount_t initial_amount;        ///< 日初可用资金
    amount_t withholding_amount;    ///< 系统预扣的资金（买入未成交冻结资金）
    amount_t fund_buy_amount;       ///< 累计买入成交金额
    amount_t fund_buy_fee;          ///< 累计买入成交费用
    amount_t fund_sell_amount;      ///< 累计卖出成交金额
    amount_t fund_sell_fee;         ///< 累计卖出成交费用
    amount_t withdraw_amount;       ///< 当日划出金额
    amount_t deposit_amount;        ///< 当日划入金额
    amount_t withdrawable_amount;    ///< 可取金额
} xlt_asset_info_t;

#pragma pack(pop)
}


