/**
* @file xlt_quote_struct.h
 * @author 上海前路有光数字科技
 * @date 2026-01-10
 * @brief 本文件定义XLT行情数据结构体
 */

#pragma once

#include "xlt_data_type.h"

namespace x1{

#pragma pack(push, 1)

/**
 * @brief 股票行情全量静态信息
 */
typedef struct xlt_static_quote_full_info_t {
    ExchangeIndex  exchange_index;							///< 交易所代码
    char               security_code[CONST_STR_SECURITY_CODE_LEN];	///< 证券代码，字符串，带'\\0'
    char               security_name[CONST_SECURITY_NAME_LEN];	///< 证券名称，字符串，带'\\0'
    QuoteSecurityType  security_type;					    ///< 合约类型
    bool is_registration;									///< 是否注册制(仅适用创业板股票，创新企业股票及存托凭证)
    bool is_vie;											///< 是否具有协议控制架构(仅适用创业板股票，创新企业股票及存托凭证)
    bool is_noprofit;										///< 是否尚未盈利(仅适用创业板股票，创新企业股票及存托凭证)
    bool is_weighted_voting_rights;							///< 是否存在投票权差异(仅适用创业板股票，创新企业股票及存托凭证)
    bool is_have_price_limit;								///< 是否有涨跌幅限制(注：不提供具体幅度，可通过涨跌停价和昨收价来计算幅度)
    int64_t upper_limit_price;								///< 涨停价（仅在有涨跌幅限制时有效）
    int64_t lower_limit_price;								///< 跌停价（仅在有涨跌幅限制时有效）
    int64_t pre_close_price;								///< 昨收价
    int64_t price_tick;										///< 价格最小变动价位
    int32_t bid_qty_upper_limit;							///< 限价买委托数量上限
    int32_t bid_qty_lower_limit;							///< 限价买委托数量下限
    int32_t bid_qty_unit;									///< 限价买数量单位
    int32_t ask_qty_upper_limit;							///< 限价卖委托数量上限
    int32_t ask_qty_lower_limit;							///< 限价卖委托数量下限
    int32_t ask_qty_unit;									///< 限价卖数量单位
    int32_t market_bid_qty_upper_limit;						///< 市价买委托数量上限
    int32_t market_bid_qty_lower_limit;						///< 市价买委托数量下限
    int32_t market_bid_qty_unit;							///< 市价买数量单位
    int32_t market_ask_qty_upper_limit;						///< 市价卖委托数量上限
    int32_t market_ask_qty_lower_limit;						///< 市价卖委托数量下限
    int32_t market_ask_qty_unit;							///< 市价卖数量单位
    char security_status[9];                                ///< 证券状态，字符串，带'\\0'
} xlt_static_quote_full_info_t;


// =====================================================================================
// 深交所 (SZSE) level1行情数据结构体
// =====================================================================================

/**
 * @brief 深交所指数快照行情数据结构体
 */
typedef struct szse_l1_index_t {
    uint16_t channel_no;                              ///< 频道代码
    int64_t orig_time;                                ///< 数据生成时间 (YYYYMMDDHHMMSSsss)
    char md_stream_id[4];                             ///< 行情类别代码（例如"900"指数等），字符串，带'\\0'
    char security_code[CONST_STR_SECURITY_CODE_LEN];  ///< 证券代码，字符串，带'\\0'
    char security_id_source[5];                       ///< 证券代码源（例如 "102" 表示深交所），字符串，带'\\0'
    char trading_phase_code[9];                       ///< 产品所处的交易阶段代码，字符串，带'\\0'
    int64_t prev_close_px;                            ///< 昨收价，放大10000倍
    int64_t num_trades;                               ///< 成交笔数
    int64_t total_volume_trade;                       ///< 成交总量，放大100倍
    int64_t total_value_trade;                        ///< 成交总金额，放大10000倍

    int64_t last_index;                              ///< 当前指数，放大1000000倍
    int64_t prev_close_index;                        ///< 昨日收盘指数，放大1000000倍
    int64_t open_index;                              ///< 开盘指数，放大1000000倍
    int64_t high_index;                              ///< 最高指数，放大1000000倍
    int64_t low_index;                               ///< 最低指数，放大1000000倍
    int64_t close_index;                             ///< 收盘指数，放大1000000倍

} szse_l1_index_t;


/**
 * @brief 深交所L1快照行情数据结构体
 */
typedef struct szse_l1_snapshot_t {
    uint16_t channel_no;                              ///< 频道代码
    int64_t orig_time;                                ///< 数据生成时间 (YYYYMMDDHHMMSSsss)
    char md_stream_id[4];                             ///< 行情类别代码（例如 "010"股票, "040"期权等），字符串，带'\\0'
    char security_code[CONST_STR_SECURITY_CODE_LEN];  ///< 证券代码，字符串，带'\\0'
    char security_id_source[5];                       ///< 证券代码源（例如 "102" 表示深交所），字符串，带'\\0'
    char trading_phase_code[9];                       ///< 产品所处的交易阶段代码，字符串，带'\\0'
    int64_t prev_close_px;                            ///< 昨收价，放大10000倍
    int64_t num_trades;                               ///< 成交笔数
    int64_t total_volume_trade;                       ///< 成交总量，放大100倍
    int64_t total_value_trade;                        ///< 成交总金额，放大10000倍

    int64_t last_px;                                  ///< 最新价，放大1000000倍
    int64_t open_px;                                  ///< 开盘价，放大1000000倍
    int64_t high_px;                                  ///< 最高价，放大1000000倍
    int64_t low_px;                                   ///< 最低价，放大1000000倍
    int64_t px_gain;                                  ///< 涨跌1（最新价减昨收价），放大1000000倍
    int64_t px_gain_2;                                ///< 涨跌2（最新价减最近价），放大1000000倍
    int64_t weighted_avg_bid_px;                      ///< 买入委托数量加权平均价，放大1000000倍
    int64_t total_bid_qty;                            ///< 买入委托总量，放大100倍
    int64_t weighted_avg_offer_px;                    ///< 卖出委托数量加权平均价，放大1000000倍
    int64_t total_offer_qty;                          ///< 卖出委托总量，放大100倍
    int64_t pe_ratio_1;                               ///< 股票市盈率1，放大1000000倍
    int64_t pe_ratio_2;                               ///< 股票市盈率2，放大1000000倍
    int64_t nav;                                      ///< T-1日基金净值，放大1000000倍，基金净值一般都为 T-1日净值；对于部分基金（如果投资境外市场基金），可能存放 T-x 日净值（x>=1，例如对于投资美股的基金，x=2）。该条目仅限于基金类产品发布。
    int64_t iopv;                                     ///< 基金实时参考净值(包括ETF的IOPV)，放大1000000倍，该条目仅限于基金类产品发布。
    int64_t warrant_premium;                          ///< 权证溢价率，放大1000000倍
    int64_t limit_up_px;                              ///< 涨停价，放大1000000倍，取值为 999999999.9999 表示无涨停
    int64_t limit_down_px;                            ///< 跌停价，放大1000000倍，对于价格可以为负数的证券，取值为-999999999.9999 表示无跌停价格限制；对于价格不可以为负数的证券，取值为证券的价格档位表示无跌停价格限制，比如对于股票填写 0.01
    int64_t option_open_qty;                          ///< 期权合约持仓量，放大100倍
    int64_t option_break_px;                          ///< 期权集中竞价交易熔断参考价，放大1000000倍

    int64_t etf_buy_amount;                           ///< ETF 申购数量, 放大100倍
    int64_t etf_buy_number;                           ///< ETF 申购笔数
    int64_t etf_sell_amount;                          ///< ETF 赎回数量, 放大100倍
    int64_t etf_sell_number;                          ///< ETF 赎回笔数

    int64_t bid_px[5];                                ///< 买入委托价，放大1000000倍
    int64_t bid_volume[5];                            ///< 买入委托量，放大100倍
    int64_t offer_px[5];                              ///< 卖出委托价，放大1000000倍
    int64_t offer_volume[5];                          ///< 卖出委托量，放大100倍
} szse_l1_snapshot_t;

/**
 * @brief 深交所L1债券快照行情数据结构体
 */
typedef struct szse_l1_snapshot_bond_t {
    uint16_t channel_no;                              ///< 频道代码
    int64_t orig_time;                                ///< 数据生成时间 (YYYYMMDDHHMMSSsss)
    char md_stream_id[4];                             ///< 行情类别代码（例如 "020"债券通用质押式回购匹配交易快照行情等），字符串，带'\\0'
    char security_code[CONST_STR_SECURITY_CODE_LEN];  ///< 证券代码，字符串，带'\\0'
    char security_id_source[5];                       ///< 证券代码源（例如 "102" 表示深交所），字符串，带'\\0'
    char trading_phase_code[9];                       ///< 产品所处的交易阶段代码，字符串，带'\\0'
    int64_t prev_close_px;                            ///< 昨收价，放大10000倍
    int64_t num_trades;                               ///< 成交笔数
    int64_t total_volume_trade;                       ///< 成交总量，放大100倍
    int64_t total_value_trade;                        ///< 成交总金额，放大10000倍

    int64_t last_px;                                  ///< 最新价，放大1000000倍
    int64_t open_px;                                  ///< 开盘价，放大1000000倍
    int64_t close_px;                                 ///< 收盘价，放大1000000倍， 如证券未收市，则不发布此条目；如证券已收市且当天有成交，则发布当日收盘价；如证券已收市且当天无成交，则发布前收盘价。
    int64_t high_px;                                  ///< 最高价，放大1000000倍
    int64_t low_px;                                   ///< 最低价，放大1000000倍

    int64_t last_px_type;                             ///< 最近价的成交方式，放大100倍，当行情类别为“410=债券现券交易快照行情”或“020=债券通用质押式回购匹配交易快照行情”时，0.01=匹配成交 0.02=协商成交 0.03=点击成交 0.04=询价成交 0.05=竞买成交
    int64_t weighted_avg_px;                          ///< 加权平均价，放大1000000倍, 质押式回购的实时成交量加权平均利率及债券现券交易成交量加权平均价，质押式回购的实时成交量加权平均利率，精确到 5 位小数。
    int64_t px_gain;                                  ///< 涨跌1（最新价减昨收价），放大1000000倍
    int64_t px_gain_2;                                ///< 涨跌2（最新价减最近价），放大1000000倍
    int64_t weighted_avg_bid_px;                      ///< 买入委托数量加权平均价，放大1000000倍
    int64_t total_bid_qty;                            ///< 买入委托总量，放大100倍
    int64_t weighted_avg_offer_px;                    ///< 卖出委托数量加权平均价，放大1000000倍
    int64_t total_offer_qty;                          ///< 卖出委托总量，放大100倍
    int64_t weighted_avg_ir_gain;                     ///< 加权平均利率涨跌BP，放大1000000倍
    int64_t weighted_avg_prev_ir;                     ///< 质押式回购的昨收盘成交量加权平均利率，放大1000000倍
    int64_t match_last_px;                            ///< 匹配成交最近成交价，放大1000000倍
    int64_t auction_volume_trade;                     ///< 匹配成交成交量，放大100倍
    int64_t auction_value_trade;                      ///< 匹配成交成交金额，放大10000倍
    char sub_trading_phase_code[6][9];                ///< 交易方式所处的交易阶段代码, 交易方式取值 1=匹配成交 2=协商成交 3=点击成交 4=询价成交 5=竞买成交

    int64_t bid_px[5];                                ///< 买入委托价，放大1000000倍
    int64_t bid_volume[5];                            ///< 买入委托量，放大100倍
    int64_t offer_px[5];                              ///< 卖出委托价，放大1000000倍
    int64_t offer_volume[5];                          ///< 卖出委托量，放大100倍
} szse_l1_snapshot_bond_t;



// =====================================================================================
// 上交所 (SSE) Level1行情数据结构体
// =====================================================================================

/**
 * @brief 上交所L1指数行情快照
 */
typedef struct sse_l1_index_t{
    uint8_t security_type;                            ///< 证券类型，1 = 股票、基金、指数及债券分销；2 = 衍生品；3 = 综合业务；12 = 债券
    uint64_t sending_time;                            ///< 交易所发送时间，格式：YYYYMMDDHHmmSSsss
    uint8_t trad_ses_mode;                            ///< 交易盘交易模式，1 = 系统测试，2 = 模拟交易，3 = 产品（正常交易）
    uint32_t trade_date;                              ///< 交易日期 (YYYYMMDD)
    uint32_t last_update_time;                        ///< 最后更新时间 (HHMMSSsss)
    char md_stream_id[6];                             ///< 行情类别，如 MD001，字符串，带'\\0'
    char security_code[CONST_STR_SECURITY_CODE_LEN];  ///< 证券代码，字符串，带'\\0'
    uint64_t prev_close_px;                           ///< 昨收盘，放大100000倍
    uint64_t total_volume_traded;                     ///< 成交数量
    uint64_t num_trades;                              ///< 成交笔数
    uint64_t total_value_traded;                      ///< 成交金额，放大100倍
    char trading_phase_code[9];                       ///< 实时阶段及标志，字符串，带'\\0'

    uint64_t last_px;                                 ///< 最新指数，放大100000倍
    uint64_t open_px;                                 ///< 今开盘指数，放大100000倍
    uint64_t close_px;                                ///< 今收盘指数，放大100000倍
    uint64_t high_px;                                 ///< 最高指数，放大100000倍
    uint64_t low_px;                                  ///< 最低指数，放大100000倍
} sse_l1_index_t;


/**
 * @brief 上交所L1快照行情
 */
typedef struct sse_l1_snapshot_t{
    uint8_t security_type;                            ///< 证券类型
    uint64_t sending_time;                            ///< 交易所发送时间，格式：YYYYMMDDHHmmSSsss
    uint8_t trad_ses_mode;                            ///< 交易盘交易模式
    uint32_t trade_date;                              ///< 交易日期 (YYYYMMDD)
    uint32_t last_update_time;                        ///< 最后更新时间 (HHMMSSsss)
    char md_stream_id[6];                             ///< 行情类别，字符串，带'\\0'
    char security_code[CONST_STR_SECURITY_CODE_LEN];  ///< 证券代码，字符串，带'\\0'
    uint64_t prev_close_px;                           ///< 昨收盘，放大100000倍
    uint64_t total_volume_traded;                     ///< 成交数量
    uint64_t num_trades;                              ///< 成交笔数
    uint64_t total_value_traded;                      ///< 成交金额，放大100倍
    char trading_phase_code[9];                       ///< 实时阶段及标志，字符串，带'\\0'

    uint64_t last_px;                                 ///< 最新成交价，放大100000倍
    uint64_t open_px;                                 ///< 今开盘价，放大100000倍
    uint64_t close_px;                                ///< 今收盘价，放大100000倍
    uint64_t settle_px;                               ///< 今结算价，放大100000倍
    uint64_t high_px;                                 ///< 当日最高成交价，放大100000倍
    uint64_t low_px;                                  ///< 当日最低成交价，放大100000倍
    uint64_t iopv;                                    ///< IOPV，放大100000倍
    uint64_t pre_iopv;                                ///< 前一日IOPV，放大100000倍
    uint64_t option_ref_px;                           ///< 期权动态参考价，放大100000倍
    uint64_t option_notional_amount;                  ///< 期权虚拟匹配数量
    uint64_t pre_settle_px;                           ///< 昨日结算价，放大100000倍
    uint64_t option_open_qty;                         ///< 期权总持仓量
    uint64_t bid_px[5];                               ///< 买入委托价，放大100000倍
    uint64_t bid_volume[5];                           ///< 买入委托量
    uint64_t offer_px[5];                             ///< 卖出委托价，放大100000倍
    uint64_t offer_volume[5];                         ///< 卖出委托量
} sse_l1_snapshot_t;

#pragma pack(pop)
}
