/**
* @file xlt_data_type.h
 * @author 上海前路有光数字科技
 * @date 2026-01-10
 * @brief 本文件定义XLT API中使用的基本数据类型和常量
 */

#pragma once

#include <cstdint>
#include <climits>

namespace x1 {
// 常量定义
constexpr uint8_t CONST_STR_SECURITY_CODE_LEN = 9; ///< 证券代码长度，含字符串终止符
constexpr uint16_t CONST_ERROR_MSG_LEN = 256; ///< 错误信息长度
constexpr uint8_t CONST_SECURITY_NAME_LEN = 33; ///< 证券名称长度
constexpr uint8_t CONST_CL_ORD_ID_LEN = 11; ///< 委托合同号长度，含字符串终止符
constexpr uint8_t CONST_REPORT_ID_LEN = 32; ///< 委托回报ID长度
constexpr uint8_t CONST_ACC_NAME_LEN = 64 + 1; ///< 账户名称长度
constexpr uint8_t CONST_ACC_PASSWD_LEN = 128 + 1; ///< 账户密码长度
constexpr uint16_t CONST_TERMINAL_INFO_LEN = 256 + 1;  ///< 终端信息的字符串长度
constexpr uint8_t CONST_API_VERSION_STR_LEN = 24 + 1;  ///< API 版本信息字符串长度
constexpr uint64_t PAGE_REQ_COUNT_MIN = 1; ///< 分页请求最小数量
constexpr uint16_t CONST_TRD_EXEC_ID_LEN = 16 + 1; ///< 成交编号长度，含字符串终止符
constexpr uint8_t CONST_FUND_ACC_NAME_LEN = 15; ///< 资金账户名称长度

// 类型定义
typedef int64_t amount_t; ///< 金额 类型
typedef int64_t price_t; ///< 价格 类型
typedef int64_t quantity_t; ///< 数量 类型

/**
* @brief API日志级别
*/
enum class ApiLogLevel : uint8_t {
    LOG_FATAL = 0,      ///<fatal级别
    LOG_ERROR = 1,      ///<error级别
    LOG_WARN = 2,       ///<warn级别
    LOG_INFO = 3,       ///<info级别
    LOG_DEBUG = 4,      ///<debug级别
    LOG_TRACE = 5,      ///<trace级别
};

/**
 * @brief 交易消息推送类型
 */
enum class TradeResumeType : uint8_t {
    RESTART = 1,        ///< 从本交易日日初开始推送
    QUICK   = 2,        ///< 从最新数据开始推送
    SPECIFY = 3,        ///< 从指定sequence开始推送
    NONE    = 99,       ///< 不推送交易消息
};


/**
 * @brief 交易所编号定义
 */
enum class ExchangeIndex : uint8_t {
    INIT = 0,    ///<初始化值或者未知
    SH   = 1,    ///< 上海证券交易所。
    SZ   = 2,    ///< 深圳证券交易所。
};


/**
 * @brief 报单价格类型
 */
enum class OrderType: uint8_t {
    NONE                       = 0, ///< 无特定类型。
    BEST_FIVE_PRICE_OR_CANCEL  = 1, ///< 市转撤（市价最优五档全额成交剩余撤销）。
    LIMIT_PRICE                = 2, ///< 限价。
    BEST_FIVE_PRICE_TO_LIMIT   = 3, ///< 市转限（市价最优五档全额成交剩余转限价）（仅上海有效）。
    SELF_BEST                  = 4, ///< 本方最优。
    COUNTER_BEST               = 5, ///< 对手方最优。
    MARKET_PRICE_OR_CANCEL     = 6, ///< 市价立即成交剩余撤销（仅深圳有效）。
    MARKET_PRICE_ALL_OR_CANCEL = 7  ///< 市价全额成交或撤销（仅深圳有效）。
};


/**
 * @brief 买卖方向
 */
enum class Side : uint8_t {
    BUY  = 1, ///< 买入(ETF申购)。
    SELL = 2  ///< 卖出(ETF赎回)。
};


/**
 * @brief 业务类型
 */
enum class BusinessType : uint8_t {
    NONE              = 0,  ///<未知类型
    CASH              = 1,  ///<现货（股票、基金、深圳可转债等）集中竞价交易
    BOND_MATCH        = 2,  ///<债券现券竞价交易
    ETF               = 3,  ///<ETF申赎业务
    ISSUE             = 4,  ///<网上发行业务
    ALLOT             = 5,  ///<配售业务
    BOND_CONVERT      = 6,  ///<债转股业务
    PLEDGE_STYLE_REPO = 7,  ///<债券质押式回购
};


/**
 * @brief 订单简要状态，API订单状态通知使用
 */
enum class OrderBriefStatus : uint8_t {
    INIT      = 1,   ///< 初始。
    SUBMITTED = 2,   ///< 已报。
    FINISHED  = 3    ///< 结束。
};


/**
 * @brief 订单状态
 */
enum class OrderStatus : uint8_t {
    INIT          = 0, ///< 初始。
    ALL_TRADED    = 1, ///< 全部成交。
    PART_TRADED   = 2, ///< 部分成交。
    PART_CANCELED = 3, ///< 部分撤单。
    NOT_TRADED    = 4, ///< 未成交。
    ALL_CANCELED  = 5, ///< 全部撤单/已撤单。
    REJECTED      = 6, ///< 废单。
    COUNT         = 7  ///< 委托状态数量。
};


/**
 * @brief 持仓证券类型
 */
enum class PositionSecurityType : uint8_t {
    NORMAL    = 0,   ///<普通持仓
    PLACEMENT = 1,   ///<配售类型的持仓，包含配股、配债等
    UNKNOWN   = 99,  ///<未知类型
};

/**
* @brief 证券类型
*/
enum class QuoteSecurityType : uint8_t{
    STOCK   = 1,      ///<股票
    FUND    = 2,      ///<基金
    BOND    = 3,      ///<债券
    OPTION  = 4,      ///<期权
    INDEX   = 5,      ///<指数
    UNKNOWN = 99      ///<未知类型
};

/**
* @brief 证券适当性枚举类型
*/
enum class QualificationType : uint8_t{
    PUBLIC       = 1,	///<包括普通投资者、专业投资者 在内的所有投资者
    COMMON       = 2,	///<仅专业投资者
    ORGANIZATION = 3,	///<仅专业投资者中的机构投资者
    UNKNOWN      = 99	///<其他
};

/**
 * @brief 行情级别：Level1/Level2
 */
enum class QuoteDataLevel : uint8_t {
    LEVEL_ALL = 0,    ///<Level1和Level2
    LEVEL_1 = 1,      ///<Level1
    LEVEL_2 = 2,      ///<Level2
};

/**
 * @brief 错误信息
 */
typedef struct {
    uint64_t error_code;                      ///< 错误代码
    uint64_t sub_error_code;                  ///< 错误子代码
    char error_msg[CONST_ERROR_MSG_LEN];      ///< 错误信息
} xlt_error_info_t;


/**
 * @brief 订单查询请求-条件查询
*/
typedef struct {
    uint64_t xid;                     ///< 需要查询的委托编号，如果为0则根据后续条件进行查询，如果不为0，则只查询指定订单，忽略后续查询条件
    char security_code[CONST_STR_SECURITY_CODE_LEN]; ///<证券代码，可以为空，如果不为空，请不带空格，并以'\\0'结尾，则默认查询时间段内的所有成交回报
    int64_t begin_time;               ///< 格式为YYYYMMDDHHMMSSsss，为0则默认当前交易日0点
    int64_t end_time;                 ///< 格式为YYYYMMDDHHMMSSsss，为0则默认当前时间
    uint8_t query_unfinished;         ///< 是否只查询未完成订单：1是，0否
} xlt_order_query_param_t;

/**
 * @brief 分页查询请求参数
 */
typedef struct xlt_query_by_page_param_t {
    uint64_t req_count;              ///< 分页查询条数，若填写小于1的值，则默认查询1条
    uint64_t reference;              ///< 上一次收到的查询结果中带回来的索引，如果是从头查询，请置0
} xlt_query_by_page_param_t;

}

