/**
 * @file xlt_api.h
 * @author 上海前路有光数字科技
 * @date 2026-01-10
 * @brief 本接口定义XLT的主要接口类，包括交易和行情回调接口类。
 */

#pragma once

#include "xlt_trade_struct.h"
#include "xlt_quote_struct.h"
#include "xlt_api_config.h"
#include "sse_hpf_define.h"
#include "sze_hpf_define.h"

namespace x1 {
    /**
     * @brief 行情回调接口
     * 此类用于接收行情数据回调，以及行情接收启动结果回调，目前仅包含Level1行情回调。若使用Level2行情，请使用EFHQuoteSpi类。
     * 用户需要继承此类，并实现相应的回调函数，然后将子类实例传递给XLTApi::initialize方法，并且XLTApi::start_quote方法启动行情接收才能接收行情数据。
     * 用户可以根据需要选择性地实现部分回调函数。
     */
    class XLTQuoteSpi {
    public:
        XLTQuoteSpi() = default;
        virtual ~XLTQuoteSpi() = default;
        /**
         * @brief 行情接收启动回调
         * @param session_id 会话id，登录时得到
         * @param error_info 发生错误时返回的错误信息，当error_info为空，或者error_info.error_id为0时，表明没有错误
         */
        virtual void on_start_quote(uint64_t session_id, xlt_error_info_t* error_info) {};
        /**
         * @brief 上交所level1指数行情推送
         * @param sse_l1_index 上交所level1指数行情
         */
        virtual void on_sse_l1_index(const sse_l1_index_t* sse_l1_index) {};
        /**
         * @brief 上交所level1快照行情推送
         * @param sse_l1_snapshot 上交所level1快照行情
         */
        virtual void on_sse_l1_snapshot(const sse_l1_snapshot_t* sse_l1_snapshot) {};
        /**
         * @brief 深交所level1指数行情推送
         * @param szse_l1_index 深交所level1指数行情
         */
        virtual void on_szse_l1_index(const szse_l1_index_t* szse_l1_index) {};
        /**
         * @brief 深交所level1快照行情推送
         * @param szse_l1_snapshot 深交所level1快照行情
         */
        virtual void on_szse_l1_snapshot(const szse_l1_snapshot_t* szse_l1_snapshot) {};
        /**
         * @brief 深交所level1债券快照行情推送
         * @param szse_l1_snapshot_bond 深交所level1债券快照行情
         */
        virtual void on_szse_l1_snapshot_bond(const szse_l1_snapshot_bond_t* szse_l1_snapshot_bond) {};
    };

    /**
     * @brief 盛立 行情回调接口
     * 此类是XLTQuoteSpi子类，包含XLTQuoteSpi回调类的所有回调函数，同时扩展了盛立Level2行情回调函数。
     * 用户若需要使用盛立Level2行情数据，需要继承此类，并实现相应的回调函数，然后将子类实例传递给XLTApi::initialize方法，并且XLTApi::start_quote方法启动行情接收才能接收盛立Level2行情数据。
     * 用户可以根据需要选择性地实现部分回调函数。
     * @remark 盛立Level2行情回调函数均以on_report_efh_开头，盛立行情数据定义请参考sse_hpf_define.h和sze_hpf_define.h头文件。
     */
    class EFHQuoteSpi : public XLTQuoteSpi {
    public:
        virtual ~EFHQuoteSpi() = default;
        /**
         * @brief 上报上交所快照数据
         * @param p_snap lev2 快照数据结构
         */
        virtual void on_report_efh_sse_lev2_snap(const sse_hpf_lev2* p_snap) {};
        /**
         * @brief 上报上交所逐笔合并消息类型
         * @param p_tick 逐笔合并结构
         */
        virtual void on_report_efh_sse_lev2_tick_merge(const sse_hpf_tick_merge* p_tick) {};
        /**
         * @brief 上报上交所指数数据
         * @param p_idx 指数数据结构
         */
        virtual void on_report_efh_sse_lev2_idx(const sse_hpf_idx* p_idx) {};
        /**
         * @brief 上报上交所债券快照消息类型
         * @param p_bond_snap	债券快照结构
         */
        virtual void on_report_efh_sse_lev2_bond_snap(const sse_hpf_bond_snap* p_bond_snap) {};
        /**
         * @brief 上报上交所债券逐笔消息类型
         * @param p_bond_tick	债券逐笔结构
         */
        virtual void on_report_efh_sse_lev2_bond_tick(const sse_hpf_bond_tick* p_bond_tick) {};
        /**
         * @brief 上报深交所lev2快照数据
         * @param p_snap	lev2快照结构
         */
        virtual void on_report_efh_sze_lev2_snap(const sze_hpf_lev2* p_snap) {};
        /**
         * @brief 上报深交所逐笔订单数据
         * @param msg_type 消息类型,SZE_LEV2_ORDER_MSG_TYPE为逐笔委托，SZE_LEV2_EXE_MSG_TYPE为逐笔成交
         * @param p_order 订单结构
         * @param p_exe	成交类型
         */
        virtual void on_report_efh_sze_lev2_tick(int msg_type, const sze_hpf_order* p_order,
                                                 const sze_hpf_exe* p_exe) {};
        /**
         * @brief 上报深交所指数数据
         * @param p_idx	指数数据结构
         */
        virtual void on_report_efh_sze_lev2_idx(const sze_hpf_idx* p_idx) {};
        /**
         * @brief 上报深交所债券快照消息类型
         * @param p_bond_snap	债券快照结构
         */
        virtual void on_report_efh_sze_lev2_bond_snap(const sze_hpf_bond_snap* p_bond_snap) {};
        /**
         * @brief 上报深交所债券逐笔订单数据
         * @param msg_type 消息类型,SZE_LEV2_BOND_ORDER_MSG_TYPE为债券逐笔委托，SZE_LEV2_BOND_EXE_MSG_TYPE为债券逐笔成交
         * @param p_order 债券订单结构
         * @param p_exe	债券成交类型
         */
        virtual void on_report_efh_sze_lev2_bond_tick(int msg_type, const sze_hpf_bond_order* p_order,
                                                      const sze_hpf_bond_exe* p_exe) {};
    };

    /**
     * @brief 交易回调类
     */
    class XLTTradeSpi {
    public:
        virtual ~XLTTradeSpi() = default;
        /**
         * @brief 断线通知
         * 用户断线之后需要重新调用login方法，建立连接并登录
         * @param session_id 会话ID。
         */
        virtual void on_disconnect(uint64_t session_id) {};
        /**
         * @brief 登录通知
         * @param session_id 本次登录生成的会话ID。如果登录失败则为0。
         * @param request_id 此消息响应函数对应的请求ID
         * @param error_info 登录请求发生错误时返回的错误信息，当error_info为空，或者error_info.error_id为0时，表明没有错误。
         */
        virtual void on_login(uint64_t session_id, uint64_t request_id, xlt_error_info_t* error_info) {};
        /**
         * @brief 订单结束通知
         *
         * 本函数用于接收订单结束通知；当订单状态变为完全成交、撤单成功、废单等终结状态时，交易系统会发送该通知。
         *
         * @param session_id 会话ID。
         * @param order_end 包含订单结束信息的数据指针。
         */
        virtual void on_order_end(uint64_t session_id, xlt_order_end_t* order_end) {};
        /**
         * @brief 新订单申报成功响应
         *
         * 此函数用于处理来自交易系统的申报成功响应。
         *
         * @param session_id 会话ID。
         * @param order_response 申报成功响应数据指针。
         */
        virtual void on_order_response(uint64_t session_id, xlt_order_response_t* order_response) {};
        /**
         * @brief 撤单成功响应
         *
         * 此函数用于处理来自交易系统的撤单成功响应。
         *
         * @param session_id 会话ID。
         * @param cancel_response 撤单成功响应数据指针。
         */
        virtual void on_cancel_response(uint64_t session_id, xlt_cancel_response_t* cancel_response) {};
        /**
         * @brief 成交回报通知
         *
         * 此函数用于处理来自交易系统的成交回报通知。
         *
         * @param session_id 会话ID。
         * @param trade_report 成交回报通知数据指针。
         */
        virtual void on_trade_report(uint64_t session_id, xlt_trade_report_t* trade_report) {};
        /**
         * @brief 委托失败通知
         *
         * 此函数用于处理来自交易系统的委托失败通知。
         * 包括内部拒单、交易所拒单、废单及撤单失败等情况。
         * @param session_id 会话ID。
         * @param order_error_response 委托失败通知数据指针。
         */
        virtual void on_order_error_response(uint64_t session_id, xlt_order_error_response_t* order_error_response) {};
        /**
         * @brief 交易所原始订单执行报告（需用户自行解码）
         *
         * 此函数用于接收原始的订单执行报告。报告内容包括但不限于订单状态、执行细节等。
         * 由于报告是原始形式，因此需要用户自行解码以获取具体信息。
         *
         * @param session_id 会话ID。
         * @param client_id 委托源标识，用户登录时自定义
         * @param xid 报单标识，通常是订单的唯一标识符。
         * @param report_id 回报标识，32字节，非字符串。
         * @param sequence 报告的序列号，通常用于标识报告的顺序，可能为0。
         * @param exchange_index 交易所索引，表示报告来自哪个交易所。
         * @param report_size 回报长度
         * @param raw_report 报告数据的指针，具体内容依赖于报告类型，需用户自行解码。
         */
        virtual void on_raw_report(uint64_t session_id, uint8_t client_id, uint64_t xid, char report_id[32],
                                   uint64_t sequence, ExchangeIndex exchange_index, uint32_t report_size, char* raw_report) {};
        /**
         * @brief 持仓查询响应
         * @param session_id 会话id，登录时得到
         * @param request_id 此消息响应函数对应的请求ID
         * @param positions 本次响应返回的持仓列表，个数由data_count决定
         * @param data_count 本次响应返回的持仓个数
         * @param is_last 此消息响应函数是否为request_id这条请求所对应的最后一个响应，当为最后一个的时候为true，如果为false，表示还有其他后续消息响应
         * @param error_info 查询账户持仓发生错误时返回的错误信息，当error_info为空，或者error_info.error_id为0时，表明没有错误
         * @remark 一个查询请求可能对应多个响应，需要快速返回，否则会堵塞后续消息，当堵塞严重时，会触发断线
         */
        virtual void on_query_positions(uint64_t session_id, uint64_t request_id, xlt_position_info_t positions[],
                                        uint32_t data_count, bool is_last, xlt_error_info_t* error_info) {};
        /**
         * @brief 资金查询响应
         * @param session_id 会话id，登录时得到
         * @param request_id 此消息响应函数对应的请求ID
         * @param assets 查询到的资金账户列表情况，个数由data_count决定
         * @param data_count 本次响应返回的资金个数
         * @param is_last 此消息响应函数是否为request_id这条请求所对应的最后一个响应，当为最后一个的时候为true，如果为false，表示还有其他后续消息响应
         * @param error_info 查询资金账户发生错误时返回的错误信息，当error_info为空，或者error_info.error_id为0时，表明没有错误
         * @remark 需要快速返回，否则会堵塞后续消息，当堵塞严重时，会触发断线
         */
        virtual void on_query_assets(uint64_t session_id, uint64_t request_id, xlt_asset_info_t assets[],
                                     uint32_t data_count, bool is_last, xlt_error_info_t* error_info) {};
        /**
         * @brief 订单查询响应
         * @param session_id 会话id，登录时得到
         * @param request_id 此消息响应函数对应的请求ID
         * @param order_infos 本次响应返回的订单信息，个数由data_count决定
         * @param data_count 本次响应返回的数据条数
         * @param is_last 此消息响应函数是否为request_id这条请求所对应的最后一个响应，当为最后一个的时候为true，如果为false，表示还有其他后续消息响应
         * @param error_info 查询订单时发生错误时，返回的错误信息，当error_info为空，或者error_info.error_id为0时，表明没有错误
         * @remark 需要快速返回，否则会堵塞后续消息，当堵塞严重时，会触发断线
         */
        virtual void on_query_orders(uint64_t session_id, uint64_t request_id, xlt_order_info_t order_infos[],
                                     uint32_t data_count, bool is_last, xlt_error_info_t* error_info) {};
        /**
         * @brief 分页请求查询报单响应
         * @param session_id 会话id，登录时得到
         * @param request_id 此消息响应函数对应的请求ID
         * @param order_infos 查询到的一个报单
         * @param data_count 本次响应返回的数据条数
         * @param req_count 分页请求的最大数量
         * @param rsp_count 分页请求的当前回报数量
         * @param query_reference 当前报单信息所对应的查询索引，需要记录下来，在进行下一次分页查询的时候需要用到
         * @param is_last 此消息响应函数是否为request_id这条请求所对应的最后一个响应，当为最后一个的时候为true，如果为false，表示还有其他后续消息响应
         * @param error_info 查询报单时发生错误时，返回的错误信息，当error_info为空，或者error_info.error_id为0时，表明没有错误
         * @remark 当order_sequence为0，表明当次查询没有查到任何记录，当is_last为true时，如果order_sequence等于req_count，那么表示还有报单，可以进行下一次分页查询，如果不等，表示所有报单已经查询完毕。一个查询请求可能对应多个响应，需要快速返回，否则会堵塞后续消息，当堵塞严重时，会触发断线。
         */
        virtual void on_query_orders_by_page(uint64_t session_id, uint64_t request_id, xlt_order_info_t order_infos[],
                                             uint64_t data_count, uint64_t req_count, uint64_t rsp_count,
                                             uint64_t query_reference, bool is_last, xlt_error_info_t* error_info) {};
        /**
         * @brief 成交回报查询响应
         * @param session_id 会话id，登录时得到
         * @param request_id 此消息响应函数对应的请求ID
         * @param trade_reports 本次响应返回的成交信息，个数由data_count决定
         * @param data_count 本次响应返回的数据条数
         * @param is_last 此消息响应函数是否为request_id这条请求所对应的最后一个响应，当为最后一个的时候为true，如果为false，表示还有其他后续消息响应
         * @param error_info 查询发生错误时，返回的错误信息，当error_info为空，或者error_info.error_id为0时，表明没有错误
         * @remark 需要快速返回，否则会堵塞后续消息，当堵塞严重时，会触发断线
         */
        virtual void on_query_trades(uint64_t session_id, uint64_t request_id, xlt_trade_report_t trade_reports[],
                                     uint32_t data_count, bool is_last, xlt_error_info_t* error_info) {};
        /**
         * @brief 查询全量静态行情信息响应
         * @param session_id 会话id，登录时得到
         * @param request_id 请求ID
         * @param static_quote_full_infos 本次响应返回的全量静态行情信息列表，个数由data_count决定
         * @param data_count 本次响应返回的数据个数
         * @param is_last 此消息响应函数是否为request_id这条请求所对应的最后一个响应，当为最后一个的时候为true，如果为false，表示还有其他后续消息响应
         * @param error_info 查询发生错误时返回的错误信息，当error_info为空，或者error_info.error_id为0时，表明没有错误
         * @remark 一个查询请求可能对应多个响应，需要快速返回，否则会堵塞后续消息，当堵塞严重时，会触发断线
         */
        virtual void on_query_static_quote_info(uint64_t session_id, uint64_t request_id,
                                                xlt_static_quote_full_info_t static_quote_full_infos[],
                                                uint32_t data_count, bool is_last, xlt_error_info_t* error_info) {};
        /**
         * @brief 查询最新上交所level1指数行情响应
         * @param session_id 会话id，登录时得到
         * @param request_id 请求ID
         * @param sse_l1_index 上交所L1指数行情快照
         * @param error_info 查询发生错误时返回的错误信息，当error_info为空，或者error_info.error_id为0时，表明没有错误
         */
        virtual void on_query_sse_l1_index(uint64_t session_id, uint64_t request_id, sse_l1_index_t* sse_l1_index,
                                           xlt_error_info_t* error_info) {};
        /**
         * @brief 查询最新上交所level1快照行情响应
         * @param session_id 会话id，登录时得到
         * @param request_id 请求ID
         * @param sse_l1_snapshot 上交所L1快照行情
         * @param error_info 查询发生错误时返回的错误信息，当error_info为空，或者error_info.error_id为0时，表明没有错误
         */
        virtual void on_query_sse_l1_snapshot(uint64_t session_id, uint64_t request_id,
                                              sse_l1_snapshot_t* sse_l1_snapshot, xlt_error_info_t* error_info) {};
        /**
         * @brief 查询最新深交所level1指数行情响应
         * @param session_id 会话id，登录时得到
         * @param request_id 请求ID
         * @param szse_l1_index 深交所指数快照行情
         * @param error_info 查询发生错误时返回的错误信息，当error_info为空，或者error_info.error_id为0时，表明没有错误
         */
        virtual void on_query_szse_l1_index(uint64_t session_id, uint64_t request_id,
                                            szse_l1_index_t* szse_l1_index, xlt_error_info_t* error_info) {};
        /**
         * @brief 查询最新深交所level1快照行情响应
         * @param session_id 会话id，登录时得到
         * @param request_id 请求ID
         * @param szse_l1_snapshot 深交所L1快照行情
         * @param error_info 查询发生错误时返回的错误信息，当error_info为空，或者error_info.error_id为0时，表明没有错误
         */
        virtual void on_query_szse_l1_snapshot(uint64_t session_id, uint64_t request_id,
                                               szse_l1_snapshot_t* szse_l1_snapshot, xlt_error_info_t* error_info) {};
        /**
         * @brief 查询最新深交所level1债券快照行情响应
         * @param session_id 会话id，登录时得到
         * @param request_id 请求ID
         * @param szse_l1_snapshot_bond 深交所L1债券快照行情
         * @param error_info 查询发生错误时返回的错误信息，当error_info为空，或者error_info.error_id为0时，表明没有错误
         */
        virtual void on_query_szse_l1_snapshot_bond(uint64_t session_id, uint64_t request_id,
                                                    szse_l1_snapshot_bond_t* szse_l1_snapshot_bond,
                                                    xlt_error_info_t* error_info) {};
        /**
         * @brief 查询超时通知
         * @param session_id 会话id，登录时得到
         * @param request_id 超时请求ID
         */
        virtual void on_request_timeout(uint64_t session_id, uint64_t request_id) {};
        /**
         * @brief 订单超时通知
         * @param session_id 会话id，登录时得到
         * @param xid 超时的订单标识
         * @remark 订单超时通知是指在指定时间内未收到订单的成交回报或状态更新时触发的通知，不代表委托失败，需要用户自行查询确认订单状态。
         */
        virtual void on_order_timeout(uint64_t session_id, uint64_t xid) {};
    };

    /**
     * @brief 接口类
     */
    class XLTApi {
    public:
        /**
         * @brief 创建实例（只能创建一次）
         * @return 返回交易接口实例
         */
        static XLTApi* create();
        /**
         * @brief 获取API版本号
         * @return API版本号
         */
        virtual const char* get_api_version() = 0;
        /**
         * @brief 获取错误信息
         * @return 错误信息
         */
        virtual xlt_error_info_t get_last_error() = 0;
        /**
         * @brief 初始化API，登录之前必须调用此函数，只能初始化一次
         * @config 配置参数
         * @param spi 派生自交易响应类的实例，请在登录之前设定
         * @return 初始化是否成功
         */
        virtual bool initialize(XLTApiConfig* config, XLTTradeSpi* trade_spi, XLTQuoteSpi* quote_spi) = 0;
        /**
         * 用户登录请求
         * @brief 用户登请求
         * @param request_id 用于用户定位请求响应的ID，由用户自定义
         * @param client_id 客户端标识ID
         * @param account_name 交易账号
         * @param password 登录密码
         * @param pub_key 用户公钥(长度64字节)，用户开户时申请得到
         * @return 查询请求发送是否成功，true表示成功，false表示出错，此时用户可以调用get_last_error()来获取错误信息。
         */
        virtual bool login(uint64_t request_id, uint16_t client_id, const char* account_name, const char* password,
                           const char* pub_key) = 0;
        /**
         * 登出请求
         * @brief 登出请求
         * @param session_id 会话id，登录时得到
         * @return true表示成功，false表示出错，此时用户可以调用get_last_error()来获取错误信息。
         */
        virtual bool logout(uint64_t session_id) = 0;
        /**
         * @brief 获取当前交易日。只有登录成功后,才能得到正确的交易日。
         * @return 返回日期，格式为。YYYYMMDD
         */
        virtual uint32_t get_trading_day() = 0;
        /**
         * @brief 获取账户索引，只有登录成功后，才能得到正确的资金账户索引
         * @param session_id 会话id，登录时得到
         * @return 账户索引，0表示失败，非0表示成功
         */
        virtual uint64_t get_account_index(uint64_t session_id) = 0;
        /**
         * @brief 请求推送交易数据
         * 每个session_id，仅第一次调用有效。
         * @param session_id 资金账户对应的session_id，登录时得到
         * @param resume_type 数据（订单回报等）推送方式
         * @param start_sequence 数据推送的起始序列号，默认值为0，当resume_type为TradeResumeType::SPECIFY时有效
         * @return true表示成功，false表示出错，此时用户可以调用get_last_error()来获取错误信息。
         */
        virtual bool subscribe_report(uint64_t session_id, TradeResumeType resume_type, uint64_t start_sequence) = 0;
        /**
         * @brief 本地发送初始化函数，预分配本地发送资源，非必须调用。
         * 若不调用，会在首次发送订单时自动进行初始化，故建议在用户线程调用发送接口前调用此函数以避免首次发送订单时的延迟。
         */
        virtual void init_local_send() = 0;
        /**
         * @brief 根据订单ID获取订单序列号
         * @param xid 订单ID
         * @return 返回订单序列号
         * @remark 若用户启用自定义订单序号功能（XLTApiConfig::set_order_seq_self_define），则此接口将返回用户自定义的订单序号xlt_order_insert_info_t.order_sequence，否则返回系统生成的订单序号。
         */
        static uint64_t get_order_sequence(uint64_t xid);
        /**
         * 报单录入请求，所有业务统一报单接口，根据业务类型字段区分不同业务。
         * Xlight接收订单后，会在报单响应函数on_order_response, on_cancel_response, on_order_error_response中返回,
         * 之后所有的成交明细会在on_trade_report中返回，当订单到达终止状态会触发订单状态结束响应。
         * @brief 委托
         * @param session_id   资金账户对应的session_id，登录时得到
         * @param order 委托信息
         * @param algo_mid 如果是普通交易账号填0；如果是算法厂商，则填写母单ID。
         * @return 返回系统生成的订单ID，如果为‘0’表示订单发送失败，非"0"表示订单发送成功，用户需要记录下返回的order_xid，它保证一个交易日内唯一，不同的交易日不保证唯一性。
         */
        virtual uint64_t insert_order(uint64_t session_id, xlt_order_insert_info_t* order, uint64_t
                                      algo_mid = 0) = 0;
        /**
         * 撤单请求。
         * 如果撤单成功，会在报单响应函数on_order_response里返回原单部撤或者全撤的消息，
         * 如果不成功，会在on_order_cancel_error响应函数中返回错误原因。
         * @brief 撤单
         * @param session_id 会话id，登录时得到
         * @param cancel_order 撤单信息
         * @param algo_mid 如果是普通交易账号填0；如果是算法厂商，则填写母单ID。
         * @return  撤单xid(在xLight系统中的ID),如果为‘0’表示撤单发送失败，非"0"表示撤单发送成功，用户需要记录下返回的order_cancel_xid，它保证一个交易日内唯一，不同的交易日不保证唯一性。
         */
        virtual uint64_t cancel_order(uint64_t session_id, xlt_cancel_order_info_t* cancel_order,
                                      uint64_t algo_mid = 0) = 0;
        /**
         * @brief 重发订单请求
         * @param session_id 会话id，登录时得到
         * @param xid 需要重发的订单ID
         * @return 请求发送是否成功，true表示成功，false表示出错，此时用户可以调用get_last_error()来获取错误信息。
         * @remark 当委托超时时，用户可以调用此函数来重发订单请求，xid为订单ID。
         */
        virtual bool resend_order_request(uint64_t session_id, uint64_t xid) = 0;
        /**
         * @brief 请求查询资产(暂未启用)
         * @param session_id 会话id，登录时得到
         * @param request_id 用于用户定位查询响应的ID，由用户自定义
         * @param algo_mid 如果是普通交易账号填0；如果是算法厂商，则填写母单ID。
         * @return 请求发送是否成功，true表示成功，false表示出错，此时用户可以调用get_last_error()来获取错误信息。
         */
        virtual bool query_assets(uint64_t session_id, uint64_t request_id, uint64_t algo_mid = 0) = 0;
        /**
         * @brief 请求查询投资者持仓(暂未启用)
         * @param session_id 会话id，登录时得到
         * @param request_id 用于用户定位查询响应的ID，由用户自定义
         * @param security_code 需要查询持仓的合约代码，可以为空，如果不为空，请不带空格，并以'\\0'结尾，表示查询全市场，如果不为空，请不带空格
         * @param exchange_index 需要查询持仓的合约所在市场，默认为INIT。如果想正确查询指定持仓，请指定exchange_index
         * @param algo_mid 如果是普通交易账号填0；如果是算法厂商，则填写母单ID。
         * @return 请求发送是否成功，true表示成功，false表示出错，此时用户可以调用get_last_error()来获取错误信息。
         * @remark 该方法如果用户提供了合约代码，则会查询此合约的持仓信息，如果合约代码为空，则默认查询所有持仓信息。
         */
        virtual bool query_positions(uint64_t session_id, uint64_t request_id, char* security_code,
                                     ExchangeIndex exchange_index = ExchangeIndex::INIT,
                                     uint64_t algo_mid = 0) = 0;
        /**
         * @brief 请求查询订单(暂未启用)
         * @param session_id 会话id，登录时得到
         * @param request_id 用于用户定位查询响应的ID，由用户自定义
         * @param order_query_param 需要查询的订单相关筛选条件，其中xid如果不为0，则只查询指定订单，否则根据后续条件进行查询：
         * 其中证券代码可以为空，则默认所有存在的证券代码，如果不为空，请不带空格,
         * 其中起始时间格式为YYYYMMDDHHMMSSsss，为0则默认当前交易日0点，结束时间格式为YYYYMMDDHHMMSSsss，为0则默认当前时间
         * @param algo_mid 如果是普通交易账号填0；如果是算法厂商，则填写母单ID。
         * @return 请求发送是否成功，true表示成功，false表示出错，此时用户可以调用get_last_error()来获取错误信息。
         */
        virtual bool query_orders(uint64_t session_id, uint64_t request_id, const xlt_order_query_param_t
                                  * order_query_param, uint64_t algo_mid = 0) = 0;
        /**
         * @brief 分页请求查询报单
         * @param session_id 会话id，登录时得到
         * @param request_id 用于用户定位查询响应的ID，由用户自定义
         * @param page_query_param 需要分页查询订单的条件，如果第一次查询，那么page_query_param.reference填0
         * @param algo_mid 如果是普通交易账号填0；如果是算法厂商，则填写母单ID。
         * @return 请求发送是否成功，true表示成功，false表示出错，此时用户可以调用get_last_error()来获取错误信息。
         * @remark 该方法支持分页查询，注意用户需要记录下最后一笔查询结果的reference以便用户下次查询使用
         */
        virtual bool query_orders_by_page(uint64_t session_id, uint64_t request_id, const xlt_query_by_page_param_t
                                          * page_query_param, uint64_t algo_mid = 0) = 0;
        /**
         * @brief 请求查询成交回报
         * @param session_id 会话id，登录时得到
         * @param request_id 用于用户定位查询响应的ID，由用户自定义
         * @param xid 需要查询的订单ID
         * @param algo_mid 如果是普通交易账号填0；如果是算法厂商，则填写母单ID。
         * @return 查询请求发送是否成功，"0"表示成功，非"0"表示出错，此时用户可以调用get_last_error()来获取错误信息
         */
        virtual bool query_trades(uint64_t session_id, uint64_t request_id, uint64_t xid, uint64_t algo_mid = 0) = 0;
        /**
         * @brief 查询静态行情信息请求
         * @param session_id 会话id，登录时得到
         * @param request_id 用于用户定位查询响应的ID，由用户自定义
         * @return 请求发送是否成功，true表示成功，false表示出错，此时用户可以调用get_last_error()来获取错误信息。
         * @remark 此接口不需要启动行情接收即可查询，查询结果将在XLTTradeSpi中回调on_query_static_quote_info
         */
        virtual bool query_static_quote_info(uint64_t session_id, uint64_t request_id) = 0;
        /**
         * @brief 查询最新上交所level1指数行情
         * @param session_id 会话id，登录时得到
         * @param request_id 用于用户定位查询响应的ID，由用户自定义
         * @param security_code 查询代码
         * @return 请求发送是否成功，true表示成功，false表示出错，此时用户可以调用get_last_error()来获取错误信息。
         * @remark 此接口不需要启动行情接收即可查询，查询结果将在XLTTradeSpi中回调on_query_sse_l1_index
         */
        virtual bool query_sse_l1_index(uint64_t session_id, uint64_t request_id, const char* security_code) = 0;
        /**
         * @brief 查询最新上交所level1快照行情
         * @param session_id 会话id，登录时得到
         * @param request_id 用于用户定位查询响应的ID，由用户自定义
         * @param security_code 查询代码
         * @return 请求发送是否成功，true表示成功，false表示出错，此时用户可以调用get_last_error()来获取错误信息。
         * @remark 此接口不需要启动行情接收即可查询，查询结果将在XLTTradeSpi中回调on_query_sse_l1_snapshot
         */
        virtual bool query_sse_l1_snapshot(uint64_t session_id, uint64_t request_id, const char* security_code) = 0;
        /**
         * @brief 查询最新深交所level1指数行情
         * @param session_id 会话id，登录时得到
         * @param request_id 用于用户定位查询响应的ID，由用户自定义
         * @param security_code 查询代码
         * @return 请求发送是否成功，true表示成功，false表示出错，此时用户可以调用get_last_error()来获取错误信息。
         * @remark 此接口不需要启动行情接收即可查询，查询结果将在XLTTradeSpi中回调on_query_szse_l1_index
         */
        virtual bool query_szse_l1_index(uint64_t session_id, uint64_t request_id, const char* security_code) = 0;
        /**
         * @brief 查询最新深交所level1快照行情
         * @param session_id 会话id，登录时得到
         * @param request_id 用于用户定位查询响应的ID，由用户自定义
         * @param security_code 查询代码
         * @return 请求发送是否成功，true表示成功，false表示出错，此时用户可以调用get_last_error()来获取错误信息。
         * @remark 此接口不需要启动行情接收即可查询，查询结果将在XLTTradeSpi中回调on_query_szse_l1_snapshot
         */
        virtual bool query_szse_l1_snapshot(uint64_t session_id, uint64_t request_id, const char* security_code) = 0;
        /**
         * @brief 查询最新深交所level1债券快照行情
         * @param session_id 会话id，登录时得到
         * @param request_id 用于用户定位查询响应的ID，由用户自定义
         * @param security_code 查询代码
         * @return 请求发送是否成功，true表示成功，false表示出错，此时用户可以调用get_last_error()来获取错误信息。
         * @remark 此接口不需要启动行情接收即可查询，查询结果将在XLTTradeSpi中回调on_query_szse_l1_snapshot_bond
         */
        virtual bool query_szse_l1_snapshot_bond(uint64_t session_id, uint64_t request_id, const char* security_code) = 0;
        /**
         * @brief 订阅指定代码行情
         * @param exchange_index 上交所或者深交所
         * @param security_code 代码
         * @return 请求发送是否成功，true表示成功，false表示出错，此时用户可以调用get_last_error()来获取错误信息。
         * @remark 建议在接收行情推送(start_quote)前调用。每次添加一只代码，可以重复调用
         */
        virtual bool subscribe_quote(ExchangeIndex exchange_index, char* security_code) =0;
        /**
         * @brief 订阅全市场代码行情
         * @param exchange_index 上交所或者深交所
         * @return 请求发送是否成功，true表示成功，false表示出错，此时用户可以调用get_last_error()来获取错误信息。
         * @remark 建议在接收行情推送(start_quote)前调用。
         */
        virtual bool subscribe_all_quote(ExchangeIndex exchange_index) =0;
        /**
         * @brief 启动行情接收
         * @param session_id  会话id，登录时得到。
         * @return 请求发送是否成功，true表示成功，false表示出错，此时用户可以调用get_last_error()来获取错误信息。
         * @remark 此方法会判断用户行情权限，若config中配置了上海或深圳level2行情，会判断此session_id对应的账号是否有level2权限。on_start_quote回调将返回启动结果。
         * 启动之后，行情组播接收将在XLTQuoteSpi中回调。
         */
        virtual bool start_quote(uint64_t session_id) = 0;

        /**
         * @brief 停止行情接收
         * @return 请求发送是否成功，true表示成功，false表示出错，此时用户可以调用get_last_error()来获取错误信息。
         * @remark 停止之后，将不再接收行情推送。
         */
        virtual bool stop_quote() = 0;

    protected:
        virtual ~XLTApi() = default;
    };
}
